//*************************************************************************************************
//
//	Description:
//		AmbientShadow.fx - Default shader for false ambient shadow rendering.
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Barry Irvine
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		BIrvine         05/18/2007  0.1           Created
//		BIrvine         05/23/2007  0.2           Refactored for new material override system
//		TNettleship     07/24/2007  0.3           Made sure samplers aren't using anisotropic filtering.
//		BIrvine         07/26/2007  0.4           Added biasing to counter errors from use of clipping planes.
//		TNettleship			10/23/2007	0.5						Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
// TODO:
// * Add Alpha Support
// * Add Fog Support
// * Try Normal Based Falloff
//*************************************************************************************************

#include "stddefs.fxh"

// Tweak me to change the intensity of the ambient shadowing effect. Lower values == paler shadows.
// Valid range: 0.0 (no shadows) ... 1.0 (max strength shadows). Setting it to values outside this range will produce silly results.
#if defined( _PS3_ )
#define TWEAKABLE_INTENSITY_REDUCTION	 0.975f
#else
#define TWEAKABLE_INTENSITY_REDUCTION	 0.75f
#endif

//-----------------------------------------------------------------------
//
// Input parameters
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


float3 gAmbientShadowColour : AMBIENT_SHADOW_COLOUR
<
	string UIWidget = "None";
  bool appEdit = true;
>;

float4x4 gAmbientShadowMatrix : AMBIENT_SHADOW_MATRIX
<
	string UIWidget = "None";
  bool appEdit = true;
>;

texture tAmbientShadowTexture : AMBIENT_SHADOW_TEXTURE
<
    bool appEdit = true;
>;

float gAmbientShadowBias : AMBIENT_SHADOW_BIAS
<
    bool appEdit = true;
>;

sampler2D sAmbientShadow : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="tAmbientShadowTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Border";
	string AddressV  = "Border";
	int BorderColor = 0;
> 
= sampler_state
{
	Texture = < tAmbientShadowTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter		= _MINFILTER;
	MagFilter		= Linear;
	MipFilter		= Linear;
	AddressU		=	Border;
	AddressV		=	Border;
#ifdef _PS3_
	PS3BorderColor	= 0x00000000;
#else
	BorderColor =	0x00000000;
#endif	
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------

// Application to Vertex
struct A2V
{
	float3	Position		: POSITION;
#if !defined(_PS3_)	
	float4	Normal			: NORMAL;
#endif
};

// Vertex to Pixel
struct V2P
{
	float4	Position		: POSITION;			// NOTE: This can't be used as an input in the pixel shader!
	float4	TexCoord		: TEXCOORD0;
#if !defined(_PS3_)	
	float4	Normal			: TEXCOORD1;
#endif
};

// Pixel to Target
struct P2T
{
	COLOUR_OUTPUT_TYPE	Colour0			: COLOR0;
};


//-----------------------------------------------------------------------
// fxc /T vs_3_0 /Gfp /E AmbientShadowVertexShader /Fc _output.fx ambientshadow.fx
V2P AmbientShadowVertexShader( A2V _Input )
{
	V2P _Output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Calculate clip-space position of the vertex
	_Output.Position = mul( LPFLOAT4( _Input.Position, Rv(1.0) ), worldviewproj );

	// Bias the Z coordinate.
	//
	// This is needed because the use of clip planes to improve fill rate slightly alters the position of the vertices, causing Z fighting artifacts.
	_Output.Position.z += gAmbientShadowBias;

	// Pass through the normal
#if !defined(_PS3_)
	_Output.Normal = _Input.Normal;
#endif	

	// Calculate Ambient Shadow Texture Coordinates
	float3 lWorldPos = mul( LPFLOAT4( _Input.Position, Rv(1.0) ), world ).xyz;
	_Output.TexCoord = mul( LPFLOAT4( lWorldPos, Rv(1.0) ), gAmbientShadowMatrix );			// TODO: merge these matrices

	return _Output;
}


//-----------------------------------------------------------------------
// fxc /T ps_3_0 /Gfp /E AmbientShadowFragmentShader /Fc _output.fx ambientshadow.fx
P2T AmbientShadowFragmentShader( V2P _Input )
{
#ifdef _PS3_
	P2T _Output;

	LPFLOAT4	lTexCoord;
	lTexCoord.xy	= (_Input.TexCoord.xy / _Input.TexCoord.w);
	
	_Output.Colour0.xyz = Rv(1) - (tex2D( sAmbientShadow, _Input.TexCoord.xy ).xyz * gAmbientShadowColour.xyz * TWEAKABLE_INTENSITY_REDUCTION);	
	_Output.Colour0.w		= Rv(0);

	return _Output;

#else

	P2T _Output;

	float4	lTexCoord;
	lTexCoord.xy	= (_Input.TexCoord.xy / _Input.TexCoord.w);
	lTexCoord.zw	= -lTexCoord.xy + 1.0f;
	clip(lTexCoord);

	// TEMP: reject pixels that aren't facing mostly upwards
	float lTemp = dot(normalize(_Input.Normal.xyz), float3(0,1,0)) - 0.1;
	//clip(lTemp);

	//

	_Output.Colour0.xyz = 1.0 - (tex2D( sAmbientShadow, _Input.TexCoord.xy ).xyz * gAmbientShadowColour.xyz * TWEAKABLE_INTENSITY_REDUCTION);
	_Output.Colour0.w		= 0;

	return _Output;
#endif	
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique ambientShadow
<
	bool supportsSpecialisedLighting = false;
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "ambientShadow";
	int    normalDeferredID		= 1;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx AmbientShadowVertexShader();
		PixelShader = compile sce_fp_rsx AmbientShadowFragmentShader();
#else
		VertexShader = compile vs_3_0 AmbientShadowVertexShader();
		PixelShader = compile ps_3_0 AmbientShadowFragmentShader();
#endif
	}
}
